<?php

namespace App\Services;

/**
 * Factory para configuraciones de plataformas de software
 *
 * Centraliza la configuración de iconos, colores y metadata para cada
 * sistema operativo soportado. Facilita agregar nuevas plataformas
 * sin modificar controladores o vistas.
 */
class PlatformConfigFactory
{
    /**
     * Colores por defecto del proyecto (variables CSS primarias)
     */
    private const DEFAULT_COLORS = [
        'colorPrimary'   => '#0B1C3F',
        'colorStrong'    => '#07132A',
        'colorWeak'      => '#162a57',
        'colorOnBrand'   => '#fff',
        'gradient'       => 'linear-gradient(135deg, #0B1C3F 0%, #07132A 100%)',
    ];

    /**
     * Configuraciones de plataformas
     * Mapea el SCodigo de la BD a configuraciones visuales
     */
    private const PLATFORM_CONFIGS = [
        'windows_desktop' => [
            'platformName'  => 'Windows',
            'platformIcon'  => 'fa-brands fa-windows',
            'platformClass' => 'windows',
            'category'      => 'desktop',
            'colorPrimary'  => '#0AA1C4',
            'colorStrong'   => '#064E60',
            'colorWeak'     => '#087A96',
            'colorOnBrand'  => '#fff',
            'gradient'      => 'linear-gradient(135deg, #0AA1C4 0%, #064E60 100%)',
            'order'         => 1,
        ],

        'android_mobile' => [
            'platformName'  => 'Android',
            'platformIcon'  => 'fa-brands fa-android',
            'platformClass' => 'android',
            'category'      => 'mobile',
            'colorPrimary'  => '#3DDC84',
            'colorStrong'   => '#2CA566',
            'colorWeak'     => '#6EE7A5',
            'colorOnBrand'  => '#fff',
            'gradient'      => 'linear-gradient(135deg, #3DDC84 0%, #2CA566 100%)',
            'order'         => 2,
        ],

        'mac_desktop' => [
            'platformName'  => 'macOS',
            'platformIcon'  => 'fa-brands fa-apple',
            'platformClass' => 'macos',
            'category'      => 'desktop',
            'colorPrimary'  => '#000000',
            'colorStrong'   => '#1C1C1E',
            'colorWeak'     => '#3A3A3C',
            'colorOnBrand'  => '#fff',
            'gradient'      => 'linear-gradient(135deg, #1C1C1E 0%, #000000 100%)',
            'order'         => 3,
        ],

        'linux_desktop' => [
            'platformName'  => 'Linux',
            'platformIcon'  => 'fa-brands fa-linux',
            'platformClass' => 'linux',
            'category'      => 'desktop',
            'colorPrimary'  => '#FCC624',
            'colorStrong'   => '#E5A400',
            'colorWeak'     => '#FFD54F',
            'colorOnBrand'  => '#000',
            'gradient'      => 'linear-gradient(135deg, #FCC624 0%, #E5A400 100%)',
            'order'         => 4,
        ],

        'ios_mobile' => [
            'platformName'  => 'iOS',
            'platformIcon'  => 'fa-brands fa-apple',
            'platformClass' => 'ios',
            'category'      => 'mobile',
            'colorPrimary'  => '#007AFF',
            'colorStrong'   => '#0051D5',
            'colorWeak'     => '#4DA2FF',
            'colorOnBrand'  => '#fff',
            'gradient'      => 'linear-gradient(135deg, #007AFF 0%, #0051D5 100%)',
            'order'         => 5,
        ],

        'web' => [
            'platformName'  => 'Web',
            'platformIcon'  => 'fa-solid fa-globe',
            'platformClass' => 'web',
            'category'      => 'web',
            'colorPrimary'  => '#4285F4',
            'colorStrong'   => '#1967D2',
            'colorWeak'     => '#8AB4F8',
            'colorOnBrand'  => '#fff',
            'gradient'      => 'linear-gradient(135deg, #4285F4 0%, #1967D2 100%)',
            'order'         => 6,
        ],
    ];

    /**
     * Obtiene la configuración de una plataforma por su código
     *
     * @param string $codigo Código de la plataforma (ej: 'windows_desktop')
     * @param bool $useDefaults Si true, usa colores default del proyecto cuando no hay config específica
     * @return array|null Configuración de la plataforma o null si no existe
     */
    public static function getPlatformConfig(string $codigo, bool $useDefaults = false): ?array
    {
        $codigo = strtolower(trim($codigo));

        // Si existe configuración específica, retornarla
        if (isset(self::PLATFORM_CONFIGS[$codigo])) {
            return self::PLATFORM_CONFIGS[$codigo];
        }

        // Si se solicitan defaults, generar configuración básica
        if ($useDefaults) {
            return self::generateDefaultConfig($codigo);
        }

        return null;
    }

    /**
     * Obtiene todas las plataformas configuradas
     *
     * @return array Array asociativo [codigo => config]
     */
    public static function getAllPlatforms(): array
    {
        return self::PLATFORM_CONFIGS;
    }

    /**
     * Obtiene plataformas filtradas por categoría
     *
     * @param string $category Categoría (desktop, mobile, web)
     * @return array Array de configuraciones filtradas
     */
    public static function getPlatformsByCategory(string $category): array
    {
        return array_filter(
            self::PLATFORM_CONFIGS,
            fn ($config) => ($config['category'] ?? '') === $category
        );
    }

    /**
     * Genera una configuración default para códigos no configurados
     * Nota los valores para ser renerizados deben contener las palabras que se muestran en los ifs...
     * en este caso la BD debe de cumplir con incluir esa parte en el texto
     * del campo SCodigo para que funcione correctamente...
     *
     * Los códigos actuales son:
     *
     * - WINDOWS_DESKTOP
     * - MAC_DESKTOP
     * - ANDROID_MOBILE
     * - IOS_MOBILE
     *
     * @param string $codigo Código de la plataforma
     * @return array Configuración con valores default
     */
    private static function generateDefaultConfig(string $codigo): array
    {
        // Intentar detectar categoría por el código
        $category = 'other';
        //Mobile
        if (str_contains($codigo, 'mobile') || str_contains($codigo, 'android') || str_contains($codigo, 'ios')) {
            $category = 'mobile';
            // Windows
        } elseif (str_contains($codigo, 'desktop') || str_contains($codigo, 'windows') || str_contains($codigo, 'mac') || str_contains($codigo, 'linux')) {
            $category = 'desktop';
            // Web
        } elseif (str_contains($codigo, 'web')) {
            $category = 'web';
        }

        // Nombre amigable basado en el código
        $platformName = ucfirst(str_replace(['_', '-'], ' ', $codigo));

        return array_merge(self::DEFAULT_COLORS, [
            'platformName'  => $platformName,
            'platformIcon'  => 'fa-solid fa-download', // Icono genérico
            'platformClass' => strtolower(preg_replace('/[^a-z0-9]/', '', $codigo)),
            'category'      => $category,
            'order'         => 999, // Baja prioridad para configs auto-generadas
        ]);
    }

    /**
     * Verifica si existe configuración para un código
     *
     * @param string $codigo Código de la plataforma
     * @return bool True si existe configuración
     */
    public static function hasConfig(string $codigo): bool
    {
        return isset(self::PLATFORM_CONFIGS[strtolower(trim($codigo))]);
    }

    /**
     * Obtiene el orden de prioridad de una plataforma
     *
     * @param string $codigo Código de la plataforma
     * @return int Orden (menor = mayor prioridad)
     */
    public static function getOrder(string $codigo): int
    {
        $config = self::getPlatformConfig($codigo);
        return $config['order'] ?? 999;
    }

    /**
     * Ordena un array de códigos de plataforma por prioridad
     *
     * @param array $codigos Array de códigos
     * @return array Códigos ordenados
     */
    public static function sortByOrder(array $codigos): array
    {
        usort($codigos, function ($a, $b) {
            return self::getOrder($a) <=> self::getOrder($b);
        });
        return $codigos;
    }
}
