{{--
    COMPONENTE REUTILIZABLE: SOFTWARE CARD

    Componente para mostrar información de software con versiones descargables

    Props:
    @param string $platformName - Nombre de la plataforma (ej: "Windows", "Android")
    @param string $platformIcon - Clase de Font Awesome para el ícono (ej: "fa-brands fa-windows")
    @param string $platformClass - Clase CSS para el tema de colores (ej: "windows", "android")
    @param string $softwareName - Nombre del software
    @param Collection $versions - Colección de versiones del software
    @param Collection $filesPerVersion - Archivos agrupados por IdVersionApp
    @param string $colorPrimary - Color primario (opcional, default: #0B1C3F)
    @param string $colorStrong - Color fuerte/oscuro (opcional, default: #07132A)
    @param string $colorWeak - Color débil/claro (opcional, default: #162a57)
    @param string $colorOnBrand - Color del texto sobre brand (opcional, default: #fff)
    @param string $gradient - Gradiente personalizado (opcional)
    @param string $documentationUrl - URL de documentación (opcional)
    @param string $videoUrl - URL del video (opcional)
    @param string $uniqueId - ID único para evitar conflictos (opcional, se genera automático)
--}}

@php
    // Valores por defecto
    $colorPrimary = $colorPrimary ?? '#0B1C3F';
    $colorStrong = $colorStrong ?? '#07132A';
    $colorWeak = $colorWeak ?? '#162a57';
    $colorOnBrand = $colorOnBrand ?? '#fff';
    $gradient = $gradient ?? "linear-gradient(135deg, {$colorPrimary} 0%, {$colorStrong} 100%)";
    $uniqueId = $uniqueId ?? uniqid('sw-');

    // RGB del color primario para efectos CSS
    $rgb = sscanf($colorPrimary, "#%02x%02x%02x");
    $colorRgb = implode(', ', $rgb);

    // Función helper para valores seguros
    $safe = function($v, $campo, $fallback = 'No disponible') {
        $val = is_object($v) ? ($v->$campo ?? null) : ($v[$campo] ?? null);
        return ($val === null || $val === '' || $val === '-1') ? $fallback : $val;
    };

    // Función para formatear fechas
    $formatFechaCorta = function ($fecha) {
        $meses = ['Ene','Feb','Mar','Abr','May','Jun','Jul','Ago','Sep','Oct','Nov','Dic'];
        try {
            $carbon = \Carbon\Carbon::parse($fecha);
            return $carbon->format('d') . ' ' . $meses[$carbon->format('n') - 1] . ' ' . $carbon->format('Y');
        } catch (\Exception $e) {
            return $fecha;
        }
    };

    // Ordenar versiones por fecha (más reciente primero)
    $sortedVersions = $versions->sortByDesc(function($v) {
        try {
            return \Carbon\Carbon::parse($safe($v, 'DFecha', '1970-01-01'));
        } catch (\Exception $e) {
            return \Carbon\Carbon::parse('1970-01-01');
        }
    });

    $latestVersion = $sortedVersions->first();
    $latestVersionNumber = $latestVersion ? $safe($latestVersion, 'SVersion', '1.0.0') : '1.0.0';
    $dropdownVersions = $sortedVersions->skip(1)->take(3);
    $hasMoreVersions = $sortedVersions->count() > 4;

    $softwareDisplayName = $safe($latestVersion ?: ($versions->first() ?? []), 'SNombreSoftware', $softwareName ?? 'NOIL - Activismo');

@endphp

<div class="software-card"
     style="--card-brand: {{ $colorPrimary }};
            --card-brand-strong: {{ $colorStrong }};
            --card-brand-weak: {{ $colorWeak }};
            --card-on-brand: {{ $colorOnBrand }};
            --card-gradient: {{ $gradient }};
            --card-brand-rgb: {{ $colorRgb }};">

    {{-- HEADER --}}
    <div class="software-card__header">
        <div class="software-card__header-content">
            <div class="d-flex justify-content-between align-items-start flex-wrap gap-3">
                <div class="flex-grow-1">
                    {{-- Icono y Título en la misma fila --}}
                    <div class="d-flex align-items-center gap-3 mb-3">
                        <div class="software-card__icon">
                            <i class="{{ $platformIcon }}"></i>
                        </div>
                        <h2 class="software-card__title mb-0">{{ $platformName }}</h2>
                    </div>

                    {{-- Selector de Versiones con Chips --}}
                    @if($sortedVersions->count() > 1)
                    <div class="version-selector">
                        <label class="version-selector__label">
                            <i class="fas fa-code-branch me-2"></i>Selecciona versión:
                        </label>
                        <div class="version-chips">
                            @foreach($sortedVersions as $index => $ver)
                                @php
                                    $verId = $safe($ver, 'IdVersionApp', 0);
                                    $verNumber = $safe($ver, 'SVersion', '1.0.0');
                                    $verDate = $safe($ver, 'DFecha', now());
                                    $verSO = $safe($ver, 'SVersionSO', 'Compatible');
                                    $verSize = $safe($ver, 'STamanio', '-');
                                    $verSummary = $safe($ver, 'SResumenVersion', 'Sin resumen disponible');
                                    $verImprovements = $safe($ver, 'SMejorasVersion', '');
                                    $verRequirements = $safe($ver, 'SRequisitos', 'Sin requisitos especificados');
                                    $verPdfUrl = $safe($ver, 'URL_PDF', '');

                                    // Obtener URL de descarga
                                    $verFiles = $filesPerVersion->get($verId, collect());
                                    $verDownloadUrl = $verFiles->first() ? $safe($verFiles->first(), 'URL_Ejecutable', '') : '';
                                @endphp
                                <button
                                    class="version-chip {{ $index === 0 ? 'version-chip--active' : '' }}"
                                    data-version-id="{{ $verId }}"
                                    data-version-number="{{ $verNumber }}"
                                    data-version-date="{{ $verDate }}"
                                    data-version-date-formatted="{{ $formatFechaCorta($verDate) }}"
                                    data-version-so="{{ $verSO }}"
                                    data-version-size="{{ $verSize }}"
                                    data-version-summary="{{ htmlspecialchars($verSummary) }}"
                                    data-version-improvements="{{ htmlspecialchars($verImprovements) }}"
                                    data-version-requirements="{{ htmlspecialchars($verRequirements) }}"
                                    data-version-pdf="{{ $verPdfUrl }}"
                                    data-version-download="{{ $verDownloadUrl }}">
                                    <i class="fas fa-tag me-1"></i>
                                    <span>v{{ $verNumber }}</span>
                                    @if($index === 0)
                                        <i class="fas fa-star ms-1" title="Última versión"></i>
                                    @endif
                                </button>
                            @endforeach
                        </div>
                    </div>
                    @else
                    {{-- Si solo hay una versión, mostrar badge simple --}}
                    <span class="software-card__version-badge mt-3">
                        <i class="fas fa-tag me-1"></i>v{{ $latestVersionNumber }}
                    </span>
                    @endif
                </div>

                @if(isset($documentationUrl) || isset($videoUrl))
                <div class="software-card__links d-flex gap-2 flex-wrap">
                    @if(isset($documentationUrl) && $documentationUrl !== '-1' && $documentationUrl !== '')
                    <a href="{{ $documentationUrl }}" target="_blank" class="software-card__link" title="Documentación">
                        <i class="fas fa-book"></i>
                        <span class="d-none d-md-inline">Documentación</span>
                    </a>
                    @endif

                    @if(isset($videoUrl) && $videoUrl !== '-1' && $videoUrl !== '')
                    <a href="#video-{{ $uniqueId }}" class="software-card__link" title="Ver video">
                        <i class="fas fa-play-circle"></i>
                        <span class="d-none d-md-inline">Video</span>
                    </a>
                    @endif
                </div>
                @endif
            </div>
        </div>
    </div>

    {{-- BODY --}}
    <div class="software-card__body">
        @php
            // Solo mostrar la información de la ÚLTIMA versión
            $latestVer = $latestVersion;
            $versionId = $safe($latestVer, 'IdVersionApp', 0);
            $versionNumber = $latestVersionNumber;
            $versionDate = $safe($latestVer, 'DFecha', now());
            $versionSO = $safe($latestVer, 'SVersionSO', 'Compatible');
            $size = $safe($latestVer, 'STamanio', '-');
            $summary = $safe($latestVer, 'SResumenVersion', 'Sin resumen disponible');
            $improvements = $safe($latestVer, 'SMejorasVersion', '');
            $requirements = $safe($latestVer, 'SRequisitos', 'Sin requisitos especificados');
            $paneId = $uniqueId . '-latest';

            // Obtener archivos de la última versión
            $files = $filesPerVersion->get($versionId, collect());
            $pdfUrl = $safe($latestVer, 'URL_PDF', '');
        @endphp

        <div class="software-card__detail" id="card-detail-{{ $uniqueId }}">
            {{-- Título y metadatos --}}
            <h3 class="software-card__detail-title" id="title-{{ $uniqueId }}">
                {{ $softwareName  }} <span class="version-number">{{ $versionNumber }}</span>
            </h3>

            <div class="software-card__detail-meta">
                <div class="software-card__meta-item">
                    <i class="fas fa-desktop"></i>
                    <span id="meta-so-{{ $uniqueId }}">{{ $versionSO }}</span>
                </div>
                <div class="software-card__meta-item">
                    <i class="fas fa-calendar-alt"></i>
                    <span id="meta-date-{{ $uniqueId }}">{{ $formatFechaCorta($versionDate) }}</span>
                </div>
                <div class="software-card__meta-item" id="meta-size-container-{{ $uniqueId }}" style="{{ ($size === '-' || $size === 'No disponible') ? 'display: none;' : '' }}">
                    <i class="fas fa-file-archive"></i>
                    <span id="meta-size-{{ $uniqueId }}">{{ $size }}</span>
                </div>
            </div>

            {{-- Resumen --}}
            <div class="software-card__detail-section" id="section-summary-{{ $uniqueId }}" style="{{ ($summary === 'Sin resumen disponible' || $summary === 'No disponible') ? 'display: none;' : '' }}">
                <h6 class="software-card__detail-section-title">
                    <i class="fas fa-info-circle me-2"></i>Resumen
                </h6>
                <div class="software-card__detail-section-content" id="content-summary-{{ $uniqueId }}">
                    {{ $summary }}
                </div>
            </div>

            {{-- Mejoras --}}
            <div class="software-card__detail-section" id="section-improvements-{{ $uniqueId }}" style="{{ ($improvements === '' || $improvements === 'No disponible' || $improvements === '-1') ? 'display: none;' : '' }}">
                <h6 class="software-card__detail-section-title">
                    <i class="fas fa-sparkles me-2"></i>Mejoras aplicadas
                </h6>
                <div class="software-card__detail-section-content" id="content-improvements-{{ $uniqueId }}">
                    {!! nl2br(e($improvements)) !!}
                </div>
            </div>

            {{-- Requisitos --}}
            <div class="software-card__detail-section" id="section-requirements-{{ $uniqueId }}" style="{{ ($requirements === 'Sin requisitos especificados' || $requirements === 'No disponible') ? 'display: none;' : '' }}">
                <h6 class="software-card__detail-section-title">
                    <i class="fas fa-clipboard-check me-2"></i>Requisitos del sistema
                </h6>
                <div class="software-card__detail-section-content" id="content-requirements-{{ $uniqueId }}">
                    {{ $requirements }}
                </div>
            </div>

            {{-- Botones secundarios --}}
            <div class="software-card__btn-group" id="btn-pdf-container-{{ $uniqueId }}" style="{{ ($pdfUrl === '' || $pdfUrl === 'No disponible' || $pdfUrl === '-1') ? 'display: none;' : '' }}">
                <a href="{{ $pdfUrl }}" id="btn-pdf-{{ $uniqueId }}" class="btn btn-outline-card-brand" target="_blank">
                    <i class="fas fa-file-pdf me-2"></i>Manual PDF
                </a>
            </div>
        </div>

        {{-- BOTÓN DE DESCARGA SIMPLE - PARTE INFERIOR DERECHA --}}
        @php
            // Obtener URL de descarga de la última versión
            $latestVersionFiles = $filesPerVersion->get($safe($latestVersion, 'IdVersionApp'), collect());
            $latestDownloadUrl = $latestVersionFiles->first() ? $safe($latestVersionFiles->first(), 'URL_Ejecutable', '') : '';
        @endphp

        <div class="software-card__download-fixed" id="download-container-{{ $uniqueId }}">
            @if($latestDownloadUrl !== '' && $latestDownloadUrl !== 'No disponible' && $latestDownloadUrl !== '-1')
            {{-- SI HAY DESCARGA DISPONIBLE --}}
            <a href="{{ $latestDownloadUrl }}" id="btn-download-{{ $uniqueId }}" class="btn btn-card-brand btn-lg" target="_blank">
                <i class="fas fa-download me-2"></i>Descargar <span id="btn-download-version-{{ $uniqueId }}">v{{ $latestVersionNumber }}</span>
            </a>
            @else
            {{-- NO HAY DESCARGA DISPONIBLE --}}
            <div class="software-card__download-unavailable">
                <button class="btn btn-secondary" disabled>
                    <i class="fas fa-ban me-2"></i>No disponible
                </button>
                <small class="text-muted d-block mt-1">
                    Archivo no disponible
                </small>
            </div>
            @endif
        </div>

        {{-- Sección de Video --}}
        @if(isset($videoUrl) && $videoUrl !== '-1' && $videoUrl !== '' && $videoUrl !== 'No disponible')
        <div class="software-card__video-section" id="video-{{ $uniqueId }}">
            <h4 class="software-card__video-title">
                <i class="fas fa-play-circle"></i>
                Video Tutorial
            </h4>
            <div class="software-card__video-container">
                @php
                    // Convertir URL de YouTube a embed
                    $embedUrl = $videoUrl;
                    if (str_contains($videoUrl, 'youtube.com/watch')) {
                        parse_str(parse_url($videoUrl, PHP_URL_QUERY), $params);
                        $embedUrl = 'https://www.youtube.com/embed/' . ($params['v'] ?? '');
                    } elseif (str_contains($videoUrl, 'youtu.be/')) {
                        $embedUrl = str_replace('youtu.be/', 'youtube.com/embed/', $videoUrl);
                    }
                @endphp
                <iframe
                    src="{{ $embedUrl }}"
                    allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture"
                    allowfullscreen>
                </iframe>
            </div>
        </div>
        @endif
    </div>
</div>

{{-- JavaScript para cambio de versiones --}}
<script>
(function() {
    const uniqueId = '{{ $uniqueId }}';
    const versionChips = document.querySelectorAll(`[data-version-id]`);

    // Función helper para escapar HTML
    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    // Función helper para convertir saltos de línea
    function nl2br(str) {
        return escapeHtml(str).replace(/\n/g, '<br>');
    }

    versionChips.forEach(chip => {
        // Solo agregar listener a los chips de esta card
        if (!chip.closest(`#card-detail-${uniqueId}`)) {
            const parentCard = chip.closest('.software-card');
            if (!parentCard) return;

            chip.addEventListener('click', function() {
                // Obtener el uniqueId de esta card específica
                const cardUniqueId = parentCard.querySelector('[id^="card-detail-"]').id.replace('card-detail-', '');

                // Remover clase activa de todos los chips de esta card
                const allChipsInCard = parentCard.querySelectorAll('.version-chip');
                allChipsInCard.forEach(c => c.classList.remove('version-chip--active'));

                // Agregar clase activa al chip seleccionado
                this.classList.add('version-chip--active');

                // Obtener datos de la versión
                const versionNumber = this.dataset.versionNumber;
                const versionDate = this.dataset.versionDateFormatted;
                const versionSo = this.dataset.versionSo;
                const versionSize = this.dataset.versionSize;
                const versionSummary = this.dataset.versionSummary;
                const versionImprovements = this.dataset.versionImprovements;
                const versionRequirements = this.dataset.versionRequirements;
                const versionPdf = this.dataset.versionPdf;
                const versionDownload = this.dataset.versionDownload;

                // Actualizar título con versión
                const titleElement = document.getElementById(`title-${cardUniqueId}`);
                if (titleElement) {
                    const versionNumberSpan = titleElement.querySelector('.version-number');
                    if (versionNumberSpan) {
                        versionNumberSpan.textContent = versionNumber;
                    }
                }

                // Actualizar metadatos
                const metaSo = document.getElementById(`meta-so-${cardUniqueId}`);
                if (metaSo) metaSo.textContent = versionSo;

                const metaDate = document.getElementById(`meta-date-${cardUniqueId}`);
                if (metaDate) metaDate.textContent = versionDate;

                // Actualizar tamaño (mostrar/ocultar si no disponible)
                const metaSize = document.getElementById(`meta-size-${cardUniqueId}`);
                const metaSizeContainer = document.getElementById(`meta-size-container-${cardUniqueId}`);
                if (metaSize && metaSizeContainer) {
                    if (versionSize && versionSize !== '-' && versionSize !== 'No disponible') {
                        metaSize.textContent = versionSize;
                        metaSizeContainer.style.display = '';
                    } else {
                        metaSizeContainer.style.display = 'none';
                    }
                }

                // Actualizar resumen
                const summarySection = document.getElementById(`section-summary-${cardUniqueId}`);
                const summaryContent = document.getElementById(`content-summary-${cardUniqueId}`);
                if (summarySection && summaryContent) {
                    if (versionSummary && versionSummary !== 'Sin resumen disponible' && versionSummary !== 'No disponible') {
                        summaryContent.textContent = versionSummary;
                        summarySection.style.display = '';
                    } else {
                        summarySection.style.display = 'none';
                    }
                }

                // Actualizar mejoras
                const improvementsSection = document.getElementById(`section-improvements-${cardUniqueId}`);
                const improvementsContent = document.getElementById(`content-improvements-${cardUniqueId}`);
                if (improvementsSection && improvementsContent) {
                    if (versionImprovements && versionImprovements !== '' && versionImprovements !== 'No disponible' && versionImprovements !== '-1') {
                        improvementsContent.innerHTML = nl2br(versionImprovements);
                        improvementsSection.style.display = '';
                    } else {
                        improvementsSection.style.display = 'none';
                    }
                }

                // Actualizar requisitos
                const requirementsSection = document.getElementById(`section-requirements-${cardUniqueId}`);
                const requirementsContent = document.getElementById(`content-requirements-${cardUniqueId}`);
                if (requirementsSection && requirementsContent) {
                    if (versionRequirements && versionRequirements !== 'Sin requisitos especificados' && versionRequirements !== 'No disponible') {
                        requirementsContent.textContent = versionRequirements;
                        requirementsSection.style.display = '';
                    } else {
                        requirementsSection.style.display = 'none';
                    }
                }

                // Actualizar botón PDF
                const pdfContainer = document.getElementById(`btn-pdf-container-${cardUniqueId}`);
                const pdfBtn = document.getElementById(`btn-pdf-${cardUniqueId}`);
                if (pdfContainer && pdfBtn) {
                    if (versionPdf && versionPdf !== '' && versionPdf !== 'No disponible' && versionPdf !== '-1') {
                        pdfBtn.href = versionPdf;
                        pdfContainer.style.display = '';
                    } else {
                        pdfContainer.style.display = 'none';
                    }
                }

                // Actualizar botón de descarga
                const downloadContainer = document.getElementById(`download-container-${cardUniqueId}`);
                if (downloadContainer) {
                    if (versionDownload && versionDownload !== '' && versionDownload !== 'No disponible' && versionDownload !== '-1') {
                        // Mostrar botón disponible
                        downloadContainer.innerHTML = `
                            <a href="${versionDownload}" id="btn-download-${cardUniqueId}" class="btn btn-card-brand btn-lg" target="_blank">
                                <i class="fas fa-download me-2"></i>Descargar <span id="btn-download-version-${cardUniqueId}">v${versionNumber}</span>
                            </a>
                        `;
                    } else {
                        // Mostrar botón no disponible
                        downloadContainer.innerHTML = `
                            <div class="software-card__download-unavailable">
                                <button class="btn btn-secondary" disabled>
                                    <i class="fas fa-ban me-2"></i>No disponible
                                </button>
                                <small class="text-muted d-block mt-1">
                                    Archivo no disponible
                                </small>
                            </div>
                        `;
                    }
                }
            });
        }
    });
})();
</script>
